#include "recorder.h"

/**
 * @brief Recorder::Recorder Constructor method, it will initialize the object
 * @param folderPath The folder where videos, audios and log will be recorded
 * @param currentAtendees People that are going to atendee at the session
 * @param currentItems Items that are provided to be discussed
 * @param sessionName A name for the session
 * @param shortDescription A brief description
 * @param parent
 */
Recorder::Recorder(int record_id, QWidget *parent)
    : QWidget(parent)
{
    if (db.isOpen()){
        QSqlQuery query;

        query.prepare("SELECT * FROM record WHERE id=:recordId");
        query.bindValue(":recordId", record_id);
        query.exec();

        query.first();
        if (query.isValid()){
            this->recordId = record_id;
            this->sessionName = query.value("izena").toString();
            this->dateTime = query.value("data").toString();
            this->shortDescription = query.value("deskribapena").toString();
            this->sessionType = query.value("mota").toString();
            this->folderPath = query.value("folderPath").toString();
            this->audioVideo = (query.value("audioVideo").toInt() == 1);

            /* PARTE HARTZAILEAK */
            this->currentAtendees.clear();

            QStringListIterator atendeesItr(query.value("atendees").toString().split("@@_@@"));
            while (atendeesItr.hasNext()){
                QString elem = atendeesItr.next();

                if (elem != "")
                    this->currentAtendees.append(elem);
            }

            /* GAIAK */
            this->currentItems.clear();

            QStringListIterator itemsItr(query.value("themes").toString().split("@@_@@"));
            while (itemsItr.hasNext()){
                QString elem = itemsItr.next();

                if (elem != "")
                    this->currentItems.append(elem);
            }

            /* FITXATEGIAK */
            this->currentFiles.clear();

            QStringListIterator filesItr(query.value("files").toString().split("@@_@@"));
            while (filesItr.hasNext()){
                QString elem = filesItr.next();

                if (elem != "")
                    this->currentFiles.append(elem);
            }
        }
    }

    //// qdebug()<<"New object";
    //QGst::init(NULL, NULL);

    this->started=false;
    this->paused=false;
    this->itemKopurua = 0;
    this->atendeeKopurua = 0;

    m_ui.setupUi(this);
    m_ui.stepsLayout->addLayout(Steps::getSteps(2));
    m_ui.videoSourceGroupBox->hide();
    m_ui.pauseButton->setEnabled(false);
    m_ui.verifyPushButton->hide();
    m_ui.notesGroupBox->setEnabled(false);

    m_ui.itemGehituBotoia->setStyleSheet("border: none;");
    m_ui.itemGehituBotoia->setCursor(Qt::PointingHandCursor);

    m_ui.atendeeGehituBotoia->setStyleSheet("border: none;");
    m_ui.atendeeGehituBotoia->setCursor(Qt::PointingHandCursor);

    QStringList goiburuak;
    goiburuak << tr("Noiz") << tr("Oharra");
    m_ui.notesZerrenda->setColumnCount(2);
    m_ui.notesZerrenda->setHorizontalHeaderLabels(goiburuak);
    m_ui.notesZerrenda->horizontalHeader()->setStretchLastSection(true);
    m_ui.notesZerrenda->horizontalHeader()->setStyleSheet("border: 1px solid #fff; border-bottom: 1px solid #ccc");

    m_ui.notesZerrenda->setEditTriggers(QAbstractItemView::NoEditTriggers);

    m_ui.atendeesLayout->setAlignment(Qt::AlignTop);
    m_ui.itemsLayout->setAlignment(Qt::AlignTop);

    this->itemActiveButton = NULL;
    this->atendeeActiveButton = NULL;

    // hemen gehitu daitezke gauzak

    setWindowTitle(tr("Recorder"));

    ///Import, initialize and create UI for all Items and Atendees
    importAtendees(this->currentAtendees);
    importItems(this->currentItems);
    loadAtendees();
    loadItems();

    QSettings settings("bideoakta", "iametza");
   // m_ui.audioSourceSetting->setText(settings.value("audio_iturria_text").toString());
    m_ui.audioSourceSetting->setText(tr("Microphone"));
    m_ui.videoSourceSetting->setText(settings.value("bideo_iturria_text").toString());
    m_ui.videoFormatSetting->setText(settings.value("bideo_formatua_text").toString());

    if (settings.value("bideo_iturria_index").toInt() == 2)
        m_ui.videoFormatSetting->setDisabled(true);

    /// Initialize timer
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(showTime()));
    timer->start(1000);

    /// Initialize timer (dB)
    QTimer *timer2 = new QTimer(this);
    connect(timer2, SIGNAL(timeout()), this, SLOT(show_dB()));
    timer2->start(100);

    m_ui.dBProgressBar->setMinimum(-1); // tiene que ser menor que el máximo para que no "se mueva" antes de empezar...
    m_ui.dBProgressBar->setMaximum(0);
    m_ui.dBProgressBar->setValue(m_ui.dBProgressBar->minimum());
    m_ui.dBProgressBar->setTextVisible(false);
}

Recorder::~Recorder()
{
    //// qdebug()<<"Destructor";
}

/**
 * @brief Recorder::start This method will start recording
 * It will initialize GStreamer pipeline with the selected input parameters:
 * Filename, filePath, audio or video, video input, video mode, etc...
 */
void Recorder::start()
{
    /// Fitxategien izena kalkulatu
    this->dateTime = QDateTime::currentDateTime().toString("yyyy/MM/dd HH:mm:ss");

    QString tmpName = this->dateTime + " " + this->sessionName;

    //remove all non alphanumeric characters
    tmpName.remove(QRegExp(QString::fromUtf8("[-`~!@#$%^&*()_—+=|:;<>«»,.?/{}\'\"\\\[\\\]\\\\]")));
    tmpName.replace( " ", "_" );

    this->fileName = tmpName;

    /// Fitxategiak gordetzeko karpeta berria sortu
    if (QDir().mkdir (this->folderPath + "/" + this->fileName))
        this->folderPath = this->folderPath + "/" + this->fileName;

    /// Init stream
    this->stream = new QGstStream();
    this->stream->setPath(this->folderPath + "/" + this->fileName);

    /// Setup video input and audio input
    QSettings settings("bideoakta", "iametza");
    int videoInputSelector = settings.value("bideo_iturria_index").toInt();
    int modeInputSelector = settings.value("bideo_formatua_index").toInt();

    /// Read video input resolution from GUI
    int mode;
    mode=modeInputSelector+1;

    /// Read video input from GUI
    int connection;
    switch(videoInputSelector){
        case 1: connection = 2; break;
        case 0: connection = 1; break;
        default: connection = 0; break;
    }

    /*if(videoInputSelector==2)
    {
        m_ui.inputModeComboBox->hide();
        this->audioVideo=false;
    }*/

    this->stream->setConnection(connection);
    this->stream->setMode(mode);

    /*// qdebug()<<"connection "<<connection;
    // qdebug()<<"mode "<<mode;*/

    if(videoInputSelector==0 || videoInputSelector==1){
        ///video/audio mode, set connection and resolution
        this->audioVideo=true;

        this->stream->setType(1);
        this->stream->setConnection(connection);
        this->stream->setMode(mode);
    }
    else{
        ///only audio
        this->stream->setType(0);
    }

    m_ui.startStopButton->setText(tr("Stop recording"));
    m_ui.startStopButton->setStyleSheet("background-color: rgb(204,0,0);"
                                        "border: 1px solid rgb(0, 0, 0);"
                                        "color: rgb(255, 255, 255)");
    m_ui.startStopButton->setIcon(QIcon(":/images/stop.png"));


    m_ui.pauseButton->setEnabled(true);
    m_ui.notesGroupBox->setEnabled(true);
    setItemButtons(true);

    this->stream->start();
    this->stream->play();

    /// Guardamos los datos en la base de datos
    if (db.isOpen()){
        QSqlQuery query;

        query.prepare("UPDATE record SET data=:data, folderPath=:folderPath, fileName=:fileName, audioVideo=:audioVideo, atendees=:atendees, themes=:themes, egoera='verify' WHERE id=:recordId");
        query.bindValue(":data", this->dateTime);
        query.bindValue(":folderPath", this->folderPath);
        query.bindValue(":fileName", this->fileName);
        query.bindValue(":audioVideo", (this->audioVideo ? 1 : 0));
        query.bindValue(":atendees", this->currentAtendees.join("@@_@@"));
        query.bindValue(":themes", this->currentItems.join("@@_@@"));
        query.bindValue(":recordId", this->recordId);
        query.exec();
    }

    /// Init log (hemen jartzea beharrezkoa da, folderPath eta fileName datu basean gorde eta gero)
    this->log.InitLog(this->recordId);
    this->started=true;
    this->paused=false;
}

/**
 * @brief Recorder::stop This method will stop Recording
 */
void Recorder::stop()
{
    QMessageBox msgBox;
    msgBox.setText(tr("Stop recording"));
    msgBox.setInformativeText(tr("Are you sure?"));
    msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
    msgBox.setDefaultButton(QMessageBox::Cancel);
    int ret = msgBox.exec();

    if(ret == QMessageBox::Ok){
        deselectAll();

        // Grabazioan pause egoeran badago "se mata paco"
        if (this->paused)
            this->resume();

        this->stream->stop();
        this->started=false;
        this->paused=false;

        this->log.CloseLog(this->recordId);

        //m_ui.startStopButton->setText(tr("Start recording"));
        m_ui.startStopButton->hide();
        m_ui.pauseButton->hide();
        m_ui.dBProgressBar->hide();
        m_ui.elapsedTimeLabel->hide();
        m_ui.verifyPushButton->show();
        m_ui.notesGroupBox->setEnabled(false);
        this->setItemButtons(false);

        m_ui.dBProgressBar->setValue(m_ui.dBProgressBar->minimum());
    }
}


/**
 * @brief Recorder::on_startStopButton_clicked This slot will manage start/stop button
 */
void Recorder::on_startStopButton_clicked()
{
    if(this->started == false)
    {
        this->start();
    }
    else if (this->started == true)
    {
        this->stop();
    }
}

/**
 * @brief Recorder::pause this will pause the recording
 */
void Recorder::pause()
{
    this->stream->pause();
    this->paused = true;

    this->deselectAll();
    this->setItemButtons(false);
    this->setAtendeeButtons(false);

    m_ui.pauseButton->setText(tr("Resume recording"));
}

/**
 * @brief Recorder::pause this will pause the recording
 */
void Recorder::resume()
{
    this->stream->play();
    this->paused = false;

    this->setItemButtons(true);

    m_ui.pauseButton->setText(tr("Pause recording"));
}


/**
 * @brief Recorder::on_pauseButton_clicked This slot will manage pause button
 */
void Recorder::on_pauseButton_clicked()
{
    if(this->paused == false){
        this->pause();
    }
    else{
        this->resume();
    }
}


/**
 * @brief Recorder::showTime This slot will show the elapsed time of the recording
 */
void Recorder::showTime()
{
    if(this->started == true){
        m_ui.elapsedTimeLabel->setText(getRecordingTime());
    }
}


/**
 * @brief Recorder::loadAtendees This method will load atendees list and draw clickable buttons for each one
 * Those buttons will mark the time when user is talking or not by beeing clicked
 */
void Recorder::loadAtendees()
{
    QListIterator<Atendee> itr(atendeesList);
    while (itr.hasNext())
    {
        Atendee currentAtendee= itr.next();
        QString name= currentAtendee.getName();

        //// qdebug()<<"name is "<<name;

        QPushButton *namePushButton=new QPushButton();
        namePushButton->setCheckable(true);
        namePushButton->setDisabled(true);
        namePushButton->setText(name);
        namePushButton->setObjectName(name);
        namePushButton->setStyleSheet("background-color: #fff;height: 40px;");
        namePushButton->setIcon(QIcon(":/images/pertsona.png"));

        m_ui.atendeesGrid->addWidget(namePushButton, this->atendeeKopurua/2, this->atendeeKopurua%2);

        ///Connect click signal from checkbox to manage Atendees
        QObject::connect(namePushButton,&QPushButton::toggled,this,&Recorder::atendeeSelected);

        this->atendeeKopurua++;
    }

    //// qdebug()<<"atendees loaded";
}

/**
 * @brief Recorder::loadItems This method will load atendees list and draw clickable buttons for each one
 * Those buttons will mark the time when user is talking or not by beeing clicked
 */
void Recorder::loadItems()
{
    QListIterator<Item> itr(itemList);
    while (itr.hasNext())
    {
        Item currentItem= itr.next();
        QString name= currentItem.getName();

        //// qdebug()<<"name is "<<name;

        QPushButton *namePushButton=new QPushButton();
        namePushButton->setCheckable(true);
        namePushButton->setDisabled(true);
        namePushButton->setText(name);
        namePushButton->setObjectName(name);
        namePushButton->setStyleSheet("background-color: #fff;height: 40px;");

        m_ui.itemsGrid->addWidget(namePushButton, this->itemKopurua/2, this->itemKopurua%2);

        ///Connect click signal from checkbox to manage Items
        QObject::connect(namePushButton,&QPushButton::toggled,this,&Recorder::itemSelected);

        this->itemKopurua++;
    }

    //// qdebug()<<"items loaded";
}

/**
 * @brief Recorder::itemSelected This slot will be activated when user click on the corresponding button
 *  of each user. It will call initItem or endItem depending on the state of the button.
 */
void Recorder::itemSelected(bool checked)
{
    // Lortu botoia
    QPushButton *button = qobject_cast<QPushButton *>(QObject::sender());
    if(button){

        //qDebug()<<"Manage items "<<checked<<" from "<< button->objectName() << checked;
        if(this->started == true){
            if(checked == true){
                if (this->itemActiveButton != NULL)
                    this->itemActiveButton->setChecked(false); // Ojete que esto "lanza" una señal y vuelve a entrar aquí (por eso no no hacemos nada más, se hace luego...)

                this->initItem(button->objectName());

                button->setStyleSheet(QLatin1String("background-color: rgb(148, 192, 38);color: rgb(0, 0, 0);height: 40px;"));

                this->itemActiveButton = button;

                // enable atendees
                this->setAtendeeButtons(true);

                // disable other themes
                //this->setOtherThemes(button->objectName(), true);
            }
            if(checked == false){
                this->endItem(button->objectName());

                button->setStyleSheet(QLatin1String("background-color: rgb(255, 255, 255);height: 40px;"));

                if (this->itemActiveButton == button){
                    this->itemActiveButton = NULL;

                    // disable atendees
                    this->setAtendeeButtons(false);
                }

                // disable other themes
                //this->setOtherThemes(button->objectName(), false);
            }
        }
    }
}

/**
 * @brief Recorder::initItem This method is a wrapper to mark the begining of an speech from an item
 */
void Recorder::initItem(QString item)
{
    QString currentTime=getRecordingTime();
    QListIterator<Item> itr(itemList);
    int index=0;
    while(itr.hasNext()){
        Item currentItem=itr.next();

        if(currentItem.getName() == item){
            //// qdebug()<<" Is the same than "<<item;
            currentItem.setInitTime(currentTime);
            currentItem.setEndTime(currentItem.getEndTime());
            itemList.replace(index,currentItem);
        }
        index++;
    }
}

/**
 * @brief Recorder::endItem This method is a wrapper to mark the end of an speech from an item saving init time and endtime in the log file
 * @param item: A QString with the name of the item
 */
void Recorder::endItem(QString item)
{
     QString currentTime=getRecordingTime();

     QListIterator<Item> itr(itemList);
     int index=0;
     while(itr.hasNext()){
         Item currentItem=itr.next();

         if(currentItem.getName() == item){
             currentItem.setInitTime(currentItem.getInitTime());
             currentItem.setEndTime(currentTime);
             itemList.replace(index,currentItem);
             writeItem(currentItem.getId(),item,currentItem.getInitTime(),currentItem.getEndTime());
         }

         index++;
     }


}

/**
 * @brief Recorder::atendeeSelected This slot will be activated when user click on the corresponding button and will call to save
 * initTime or endTime of the atendee depending on the status of the button
 */
void Recorder::atendeeSelected(bool checked)
{
    // Lortu botoia
    QPushButton *button = qobject_cast<QPushButton *>(QObject::sender());
    if(button){

        //qDebug()<<"Manage atendees "<<checked<<" from "<<button->objectName() << checked;
        if(this->started == true){
            if(checked == true){
                if (this->atendeeActiveButton != NULL)
                    this->atendeeActiveButton->setChecked(false); // Ojete que esto "lanza" una señal y vuelve a entrar aquí (por eso no no hacemos nada más, se hace luego...)

                this->initSpeaking(button->objectName());

                button->setStyleSheet(QLatin1String("background-color: rgb(148, 192, 38);height: 40px;"));

                this->atendeeActiveButton = button;
            }
            if(checked == false){
                this->endSpeaking(button->objectName());

                button->setStyleSheet(QLatin1String("background-color: rgb(255, 255, 255);height: 40px;"));

                if (this->atendeeActiveButton == button)
                    this->atendeeActiveButton = NULL;
            }
        }
    }
}

/**
 * @brief Recorder::initSpeaking This method is a wrapper to mark the begining of an speech from an atendee
 * @param speaker: A QString with the name of the speaker
 */
void Recorder::initSpeaking(QString speaker)
{
    QString currentTime=getRecordingTime();

    QListIterator<Atendee> itr(atendeesList);
    int index=0;
    while(itr.hasNext()){
        Atendee currentAtendee=itr.next();

        if(currentAtendee.getName() == speaker){
            //// qdebug()<<" Is the same than "<<speaker;
            currentAtendee.setInitTime(currentTime);
            atendeesList.replace(index,currentAtendee);
        }

        index++;
    }
    //// qdebug()<<speaker<<" started speaking at "<<currentTime;
    //printAtendeeList();
}

/**
 * @brief Recorder::endSpeaking This method is a wrapper to mark the end of an speech from an atendee and save init and ending time on the log
 * @param speaker: A QString with the name of the speaker
 */
void Recorder::endSpeaking(QString speaker)
{
     QString currentTime=getRecordingTime();
     QListIterator<Atendee> itr(atendeesList);
     int index=0;
     while(itr.hasNext()){
         Atendee currentAtendee=itr.next();

         if(currentAtendee.getName() == speaker){
             //// qdebug()<<" Is the same than "<<speaker;
             currentAtendee.setEndTime(currentTime);
             //// qdebug()<<" Init time is "<<currentAtendee.getInitTime();
             atendeesList.replace(index,currentAtendee);
             writeSpeakerSpeaking(speaker,currentAtendee.getInitTime(),currentAtendee.getEndTime());
         }

         index++;
     }
     //// qdebug()<<speaker<<" ended speaking at "<<currentTime;
     //printAtendeeList();
}

/**
 * @brief Recorder::printAtendeeList This method will print atendeeList on terminal
 * DEPRECATED
 */
void Recorder::printAtendeeList()
{
    // qdebug()<<"printAtendeeList";
    QListIterator<Atendee> itr(atendeesList);
    while(itr.hasNext()){
        Atendee currentAtendee = itr.next();
        /*// qdebug()<<"name: "<<currentAtendee.getName();
        // qdebug()<<"initTime: "<<currentAtendee.getInitTime();
        // qdebug()<<"endTime: "<<currentAtendee.getEndTime();*/
    }
}

/**
 * @brief Recorder::getRecordingTime
 * @return a QString that holds the elapsed time in hh:mm:ss format
 */
QString Recorder::getRecordingTime()
{
    QTime currentTime=QTime::fromString("00:00:00","hh:mm:ss");

    //// qdebug()<<"getRecordingTime";
    if(this->started == true){
        int millisecs = this->stream->getCurrentMillisecs();
        //// qdebug()<<"millisecs "<<millisecs;
        currentTime=QTime::fromString("00:00:00","hh:mm:ss").addMSecs(millisecs);

    }
    //// qdebug()<<currentTime.toString();
    return currentTime.toString();
}



/**
 * @brief writeSpeakerSpeaking this method will write who atendee is speaking and which time did he took to speak
 * @param speaker Speaker name
 * @param initTime Init time in hh:mm:ss format
 * @param endTime End time in hh:mm:ss
 */
void Recorder::writeSpeakerSpeaking(QString speaker,QString initTime,QString endTime)
{
    this->log.SaveLogMessage(0,speaker,initTime,endTime);

    return;
}
/**
 * @brief writeItem this method will write who item is being treated  which time did that took
 * @param id Item id
 * @param item Item name
 * @param initTime Init time in hh:mm:ss format
 * @param endTime End time in hh:mm:ss
 */
void Recorder::writeItem(QString id, QString item,QString initTime,QString endTime)
{
    this->log.SaveLogMessage(1,item,initTime,endTime,id);

    return;
}

/**
 * @brief Recorder::importAtendees This method will import and initialize list of atendees in the internal list
 * @param atendees a QStringList that contains the names of all atendees to the meeting
 */
void Recorder::importAtendees(QStringList atendees)
{

    QStringListIterator itr(atendees);
    while (itr.hasNext()){
        QString name=itr.next();
        //// qdebug()<<"importing "<<name;
        Atendee atendee;
        atendee.setName(name);
        atendee.setInitTime("NaN");
        atendee.setEndTime("NaN");
        this->atendeesList.append(atendee);
    }
}

/**
 * @brief Recorder::importItems This method will import and initialize a list of items in the internal list
 * @param atendees a QStringList that contains the names of all items to the meeting
 */
void Recorder::importItems(QStringList items)
{

    QStringListIterator itr(items);
    while (itr.hasNext()){
        QString elem=itr.next();
        //// qdebug()<<"importing "<<elem;
        Item item;
        QString id = Funtzioak::getTxurroThemeId(elem);
        QString tema = Funtzioak::getTxurroThemeIzena(elem, true);

        item.setId(id);
        item.setName(tema);
        item.setInitTime("NaN");
        item.setEndTime("NaN");
        this->itemList.append(item);
    }
}

/**
 * @brief Recorder::on_verifyPushButton_clicked This slot will load MediaApp
 */
void Recorder::on_verifyPushButton_clicked()
{
    emit mediaAppSetCentralWidget(this->recordId);
}

void Recorder::setItemButtons(bool enabled)
{
    for (int i = 0; i < m_ui.itemsGrid->count(); ++i){
        QPushButton *button = qobject_cast<QPushButton *>(m_ui.itemsGrid->itemAt(i)->widget());
        if (button){
            if (button->isChecked())
                button->toggle();

            button->setEnabled(enabled);
        }
    }
}

void Recorder::setAtendeeButtons(bool enabled)
{
    for (int i = 0; i < m_ui.atendeesGrid->count(); ++i){
        QPushButton *button = qobject_cast<QPushButton *>(m_ui.atendeesGrid->itemAt(i)->widget());
        if (button){
            if (button->isChecked())
                button->toggle();

            button->setEnabled(enabled);
        }
    }
}

void Recorder::show_dB()
{
    if(started == true){
        int dB = static_cast<int>(this->stream->get_dB());

        // Modificamos el "mínimo" de la barra si el valor actual es menor, pero siempre y cuando sea mayor que -100
        m_ui.dBProgressBar->setMinimum(std::max(-100, std::min(m_ui.dBProgressBar->minimum(), dB)));

        m_ui.dBProgressBar->setValue(dB);
    }
}

void Recorder::deselectAll()
{
    for (int i = 0; i < m_ui.atendeesGrid->count(); ++i){
       QPushButton *button = qobject_cast<QPushButton *>(m_ui.atendeesGrid->itemAt(i)->widget());

       if (button && button->isChecked()){
           button->toggle();
       }
    }

    for (int i = 0; i < m_ui.itemsGrid->count(); ++i){
       QPushButton *button = qobject_cast<QPushButton *>(m_ui.itemsGrid->itemAt(i)->widget());

       if (button && button->isChecked()){
           button->toggle();
       }
    }
}

void Recorder::setOtherThemes(QString activeThemeName, bool activeThemeEnabled)
{
    for (int i = 0; i < m_ui.itemsGrid->count(); ++i){
       QPushButton *button = qobject_cast<QPushButton *>(m_ui.itemsGrid->itemAt(i)->widget());

       if (button && button->objectName() != activeThemeName){
           button->setDisabled(activeThemeEnabled);
       }
    }
}

void Recorder::on_gehituNoteBotoia_clicked()
{
    int millisecs = this->stream->getCurrentMillisecs();
    QTime currentTime= QTime::fromMSecsSinceStartOfDay(millisecs);

    /// Guardamos los datos en la base de datos
    if (db.isOpen() && !m_ui.noteTestua->text().isEmpty()){
        QSqlQuery query;

        query.prepare("INSERT INTO record_notes (noiz, testua, record_id) "
                      "VALUES (:noiz, :testua, :record_id)");
        //query.bindValue(":noiz", millisecs);
        query.bindValue(":noiz", currentTime.toString("hh:mm:ss"));
        query.bindValue(":testua", m_ui.noteTestua->text());
        query.bindValue(":record_id", this->recordId);
        query.exec();

        /// Pintamos los datos en la tabla
        m_ui.notesZerrenda->insertRow(0);
        m_ui.notesZerrenda->setItem(0, 0, new QTableWidgetItem(currentTime.toString("hh:mm:ss")));
        m_ui.notesZerrenda->setItem(0, 1, new QTableWidgetItem(m_ui.noteTestua->text()));
    }

    m_ui.noteTestua->setText("");
}

void Recorder::on_noteTestua_returnPressed()
{
    this->on_gehituNoteBotoia_clicked();
}

void Recorder::on_itemGehituBotoia_clicked()
{
    selectThemeWidget = new selectTheme();
    selectThemeWidget->setAttribute(Qt::WA_DeleteOnClose);
    selectThemeWidget->setWindowModality(Qt::ApplicationModal);

    // Centramos la ventana
    selectThemeWidget->move(erdianKokatu(selectThemeWidget->width(), selectThemeWidget->height()));

    // Mostramos la ventana
    selectThemeWidget->show();

    connect(selectThemeWidget, SIGNAL(themesKargatu(QList<QString>)), this, SLOT(on_themesKargatu(QList<QString>)));
}

void Recorder::on_themesKargatu(QList<QString> zerrenda)
{
    bool aldaketarik = false, desgaituta = false;

    // Miramos si hay algún botón desactivado para desactivar los posibles botones nuevos
    for (int i = 0; i < m_ui.itemsGrid->count(); ++i){
      QPushButton *button = qobject_cast<QPushButton *>(m_ui.itemsGrid->itemAt(i)->widget());
      if (button && !button->isEnabled()){
        desgaituta = true;
        break;
      }
    }

    for (int i = 0; i < zerrenda.size(); ++i){
        if (!this->currentItems.contains(zerrenda.at(i))){
            aldaketarik = true;

            // Item-a gehitu
            Item item;
            QString id = Funtzioak::getTxurroThemeId(zerrenda.at(i));
            QString tema = Funtzioak::getTxurroThemeIzena(zerrenda.at(i));

            item.setId(id);
            item.setName(tema);
            item.setInitTime("NaN");
            item.setEndTime("NaN");
            this->itemList.append(item);

            // Botoia gehitu
            QPushButton *namePushButton=new QPushButton();
            namePushButton->setCheckable(true);
            namePushButton->setDisabled(desgaituta);
            namePushButton->setText(tema);
            namePushButton->setObjectName(tema);
            namePushButton->setStyleSheet("background-color: #fff;height: 40px;");

            m_ui.itemsGrid->addWidget(namePushButton, this->itemKopurua/2, this->itemKopurua%2);

            ///Connect click signal from checkbox to manageAtendees
            QObject::connect(namePushButton,&QPushButton::toggled,this,&Recorder::itemSelected);

            this->currentItems.append(zerrenda.at(i));
            this->itemKopurua++;
        }
    }

    if (aldaketarik){
        /// Guardamos los datos en la base de datos
        if (db.isOpen()){
            QSqlQuery query;
            query.prepare("UPDATE record SET themes=:themes WHERE id=:id");
            query.bindValue(":themes", this->currentItems.join("@@_@@"));
            query.bindValue(":id", this->recordId);
            query.exec();
        }
    }
}

QPoint Recorder::erdianKokatu (int zabalera, int altuera){
    QPoint puntu;

    /* Pantailaren erdian kokatzeko */
    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    puntu.setX((screenGeometry.width() - zabalera) / 2);
    puntu.setY((screenGeometry.height() - altuera) / 2);

    return (puntu);
}

void Recorder::on_atendeeGehituBotoia_clicked()
{
    selectAtendeeWidget = new selectAtendee();
    selectAtendeeWidget->setAttribute(Qt::WA_DeleteOnClose);
    selectAtendeeWidget->setWindowModality(Qt::ApplicationModal);

    // Centramos la ventana
    selectAtendeeWidget->move(erdianKokatu(selectAtendeeWidget->width(), selectAtendeeWidget->height()));

    // Mostramos la ventana
    selectAtendeeWidget->show();

    connect(selectAtendeeWidget, SIGNAL(atendeesKargatu(QList<QString>)), this, SLOT(on_atendeesKargatu(QList<QString>)));
}

void Recorder::on_atendeesKargatu(QList<QString> zerrenda)
{
    bool aldaketarik = false, desgaituta = false;

    // Miramos si hay algún botón desactivado para desactivar los posibles botones nuevos
    for (int i = 0; i < m_ui.atendeesGrid->count(); ++i){
      QPushButton *button = qobject_cast<QPushButton *>(m_ui.atendeesGrid->itemAt(i)->widget());
      if (button && !button->isEnabled()){
        desgaituta = true;
        break;
      }
    }

    for (int i = 0; i < zerrenda.size(); ++i){
        if (!this->currentAtendees.contains(zerrenda.at(i))){
            aldaketarik = true;

            // Atendee-a gehitu
            Atendee atendee;
            atendee.setName(zerrenda.at(i));
            atendee.setInitTime("NaN");
            atendee.setEndTime("NaN");
            atendeesList.append(atendee);

            // Botoia gehitu
            QPushButton *namePushButton=new QPushButton();
            namePushButton->setCheckable(true);
            namePushButton->setDisabled(desgaituta);
            namePushButton->setText(zerrenda.at(i));
            namePushButton->setObjectName(zerrenda.at(i));
            namePushButton->setStyleSheet("background-color: #fff;height: 40px;");
            namePushButton->setIcon(QIcon(":/images/pertsona.png"));

            m_ui.atendeesGrid->addWidget(namePushButton, this->atendeeKopurua/2, this->atendeeKopurua%2);

            ///Connect click signal from checkbox to manageAtendees
            QObject::connect(namePushButton,&QPushButton::toggled,this,&Recorder::atendeeSelected);

            this->currentAtendees.append(zerrenda.at(i));
            this->atendeeKopurua++;
        }
    }

    if (aldaketarik){
        /// Guardamos los datos en la base de datos
        if (db.isOpen()){
            QSqlQuery query;
            query.prepare("UPDATE record SET atendees=:atendees WHERE id=:id");
            query.bindValue(":atendees", this->currentAtendees.join("@@_@@"));
            query.bindValue(":id", this->recordId);
            query.exec();
        }
    }
}
