#include "mediaapp.h"

#include <QBoxLayout>
#include <QFileDialog>
#include <QToolButton>
#include <QLabel>
#include <QSlider>
#include <QLineEdit>
#include <QMouseEvent>
#include <QPushButton>
#include <QMediaPlayer>
#include <QVideoWidget>

#include <QDebug>


/**
 * @brief MediaApp::MediaApp Constructor of the class, it will save input parameters and
 * @param audioVideo false to only audio, true to audio+video
 * @param folderPath the folder where all the content audio/video-audio and a xml containing the description of the meeting and the people and items that were discused.
 * @param itemList a list of Item elements, with their last appearance on the meeting
 * @param atendeeList a list of Atendee elements, with their last intervention on the meeting
 * @param extendedItemList a list of Item elements, with all their appearance on the meeting
 * @param extendedAtendeeList a list of Atendee elements, with all their appearance on the meeting
 * @param sessionName The name given by user to the current session
 * @param dateTime The date of the session
 * @param shortDescription Short description of the session
 * @param parent
 *
 *
 */
MediaApp::MediaApp(int record_id, QWidget *parent)
    : QWidget(parent)
{
    QList<Atendee> *extendedAtendees =new QList<Atendee>();
    QList<Item> *extendedItems = new QList<Item>();

    if (db.isOpen()){
        QSqlQuery query;

        query.prepare("SELECT * FROM record WHERE id=:recordId");
        query.bindValue(":recordId", record_id);
        query.exec();

        query.first();
        if (query.isValid()){
            this->recordId = record_id;
            this->sessionName = query.value("izena").toString();
            this->dateTime = query.value("data").toString();
            this->shortDescription = query.value("deskribapena").toString();
            this->sessionType = query.value("mota").toString();
            this->folderPath = query.value("folderPath").toString();
            this->audioVideo = (query.value("audioVideo").toInt() == 1);
            this->fileName = query.value("fileName").toString();

            /* Extended parte-hartzaileak eta gaiak */
            Log *readLog = new Log();
            readLog->readLog(this->folderPath + "/" + this->fileName + ".xml", extendedItems, extendedAtendees);

            /// Initialize Items
            QListIterator<Item> extendedItemIterator(*extendedItems);
            while(extendedItemIterator.hasNext()){
                this->extendedItemList.append(extendedItemIterator.next());
            }

            /// Initialize Atendees
            QListIterator<Atendee> extendedAtendeeIterator(*extendedAtendees);
            while(extendedAtendeeIterator.hasNext()){
                this->extendedAtendeeList.append(extendedAtendeeIterator.next());
            }
        }

        /* PARTE-HARTZAILEAK - Datu basetik hartu */
        query.prepare("SELECT * FROM atendee");
        query.exec();

        while (query.next()){
            Atendee atendee;
            atendee.setName(query.value("izena").toString());
            this->atendeeList.append(atendee);
        }

        /* GAIAK - Datu basetik hartu */
        QString izena = QLocale().name().toStdString().compare("es_ES") == 0 ? "izena_es" : "izena";
        query.prepare("SELECT id, " + izena + " FROM theme ORDER BY " + izena + " ASC");
        query.exec();

        while (query.next()){
            Item item;
            item.setId(query.value("id").toString());
            item.setName(query.value(izena).toString());
            this->itemList.append(item);
        }

    }

    this->locked = false;

    QStringList headers;
    headers << tr("Name") << tr("Start") << tr("Duration") << tr("Start ms") << tr("End ms") << "";

    m_interventionModel = new InterventionModel(headers, extendedAtendees);
    m_proxyModel = new QSortFilterProxyModel();
    m_proxyModel->setSourceModel(m_interventionModel);

    headers << "id"; // theme IDrako
    m_itemInterventionModel = new InterventionModel(headers, extendedItems);
    m_itemProxyModel = new QSortFilterProxyModel();
    m_itemProxyModel->setSourceModel(m_itemInterventionModel);

    ///Create new audio/video player
    m_player= new QMediaPlayer;
    m_player->setNotifyInterval(200);

    connect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    connect(m_player, SIGNAL(stateChanged(QMediaPlayer::State state)), this, SLOT(onStateChanged(QMediaPlayer::State state)));

    /// Notak editatzeko zerrenda
    if (db.isOpen()){
        m_notesModel = new QSqlRelationalTableModel();
        m_notesModel->setEditStrategy(QSqlTableModel::OnManualSubmit);
        m_notesModel->setTable("record_notes");
        m_notesModel->setFilter(QString("record_id = %1").arg(record_id));

        m_notesModel->setHeaderData(m_notesModel->fieldIndex("noiz"), Qt::Horizontal, tr("Noiz"));
        m_notesModel->setHeaderData(m_notesModel->fieldIndex("testua"), Qt::Horizontal, tr("Oharra"));

        // Populate the model
        if (!m_notesModel->select()) {
            showError(m_notesModel->lastError());
            return;
        }

        m_notesZerrenda = new QTableView();
        m_notesZerrenda->setModel(m_notesModel);
        m_notesZerrenda->setColumnHidden(m_notesModel->fieldIndex("id"), true);
        m_notesZerrenda->setColumnHidden(m_notesModel->fieldIndex("record_id"), true);
        m_notesZerrenda->horizontalHeader()->setStretchLastSection(true);
        m_notesZerrenda->horizontalHeader()->setStyleSheet("border: 1px solid #fff; border-bottom: 1px solid #ccc");

        m_notesZerrenda->verticalHeader()->setVisible(false);
        m_notesZerrenda->setEditTriggers(QAbstractItemView::NoEditTriggers);
        m_notesZerrenda->setSelectionBehavior(QAbstractItemView::SelectRows);
        m_notesZerrenda->setSelectionMode(QAbstractItemView::SingleSelection);
        m_notesZerrenda->setSortingEnabled(false);
        m_notesZerrenda->sortByColumn(1, Qt::AscendingOrder);
        connect(m_notesZerrenda, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onNotesZerrendaDoubleClicked(QModelIndex)));
    }

    /// Create the UI
    QVBoxLayout *appLayout = new QVBoxLayout;
    createUI(appLayout);
    setLayout(appLayout);

    m_proxyModel->sort(1); // sort by start time
    m_interventionTree->setModel(m_proxyModel);
    m_interventionTree->setAutoExpandDelay(0);
    m_interventionTree->hideColumn(3);
    m_interventionTree->hideColumn(4);
    m_interventionTree->setRootIsDecorated(false);
    m_interventionTree->header()->setStretchLastSection(false);
    m_interventionTree->header()->setSectionResizeMode(0, QHeaderView::Stretch);
    m_interventionTree->setColumnWidth(5, 20);

    m_itemProxyModel->sort(1); // sort by start time
    m_itemInterventionTree->setModel(m_itemProxyModel);
    m_itemInterventionTree->hideColumn(3);
    m_itemInterventionTree->hideColumn(4);
    m_itemInterventionTree->hideColumn(6);
    m_itemInterventionTree->setRootIsDecorated(false);
    m_itemInterventionTree->header()->setStretchLastSection(false);
    m_itemInterventionTree->header()->setSectionResizeMode(0, QHeaderView::Stretch);
    m_itemInterventionTree->setColumnWidth(5, 20);

    //onStateChanged(); //set the controls to their default state

    setWindowTitle(tr("Bideoakta player"));

    /// Set up audio/video player
    QUrl url;

    if (this->audioVideo == true){
        resize(400, 400);
        url = QUrl::fromLocalFile(this->folderPath + "/" + this->fileName + ".mp4");
    }
    else{
        //resize(1, 1);
        url = QUrl::fromLocalFile(this->folderPath + "/" + this->fileName + ".mp3");
    }

    m_player->setMedia(url);
    m_player->play();

    /// Start timed function to check if Atendee or Item has to show if is in its current time if system if stopped
    /// Workaround to show a Atendee or Item speaking in the GUI if system is not playing
    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(checkButtons()));
    timer->start(100); //time specified in ms
}

/**
 * @brief MediaApp::~MediaApp Destructor It will remove the player and liberate the file that has been played.
 * It writes the final log
 */
MediaApp::~MediaApp()
{
    //this->writeCompleteLogFromTree();
    delete m_player;
}

/**
 * @brief MediaApp::checkButtons This method will check if playing is in pause mode and system is not locked and will call
 * checkItemAtendee method to show if an Item or an Atendee is active (speaking) in the current time
 */
void MediaApp::checkButtons()
{
    //// qdebug()<<"checking";
    if(m_player->state()==QMediaPlayer::PausedState && this->locked==false)
    {
        //// qdebug()<<"Not locked";
        checkItemAtendee();
    }
}


/**
 * @brief MediaApp::onStateChanged Method to set the buttons to the appropriate status
 * @param state
 */
void MediaApp::onStateChanged(QMediaPlayer::State state)
{
    QMediaPlayer::State newState = m_player->state();
    m_playButton->setEnabled(newState != QMediaPlayer::PlayingState);
    m_pauseButton->setEnabled(newState == QMediaPlayer::PlayingState);
    m_stopButton->setEnabled(newState != QMediaPlayer::StoppedState);
    m_positionSlider->setEnabled(newState != QMediaPlayer::StoppedState);
    m_volumeSlider->setEnabled(newState != QMediaPlayer::StoppedState);
    m_volumeLabel->setEnabled(newState != QMediaPlayer::StoppedState);
    m_volumeSlider->setValue(m_player->volume());

    //if we are in Null state, call onPositionChanged() to restore
    //the position of the slider and the text on the label
    if (newState == QMediaPlayer::StoppedState) {
        onPositionChanged(0);
    }
}

//
/**
 * @brief MediaApp::onPositionChanged This slot is called when the positionChanged() signal is received from the player
 * It will display the position slider to the appropriate possition.
 * Each time the positions slider has moved the system will check if the items or atendees are speaking and their buttos must be showed pushed
 * @param position given in milliseconds
 */
void MediaApp::onPositionChanged(qint64 position)
{

    QTime length(0,0);
    QTime curpos(0,0);


    if (m_player->state() == QMediaPlayer::PlayingState) {
        length = length.addMSecs((int)m_player->duration());
        curpos = curpos.addMSecs(position);

        checkItemAtendee();
        checkItemAtendeeList();

        //m_positionLabel->setText(curpos.toString("hh:mm:ss.zzz") + " / " + length.toString("hh:mm:ss.zzz"));
        m_positionLabel->setText(curpos.toString("hh:mm:ss") + " / " + length.toString("hh:mm:ss"));
    }

    if (length != QTime(0,0)) {
        m_positionSlider->setValue(curpos.msecsTo(QTime(0,0)) * 1000 / length.msecsTo(QTime(0,0)));
    } /*else {
        ///TODO
        //m_positionSlider->setValue(0);
    }*/

    if (curpos != QTime(0,0)) {
        m_positionLabel->setEnabled(true);
        m_positionSlider->setEnabled(true);
    }
}

void MediaApp::onInterventionTreeDoubleClicked(QModelIndex index)
{
    QModelIndex proxyIndex;
    InterventionItem *item;
    // gelditu stream
    m_player->pause();

    QObject *sender = QObject::sender();
    QTreeView *senderTree = qobject_cast<QTreeView*>(sender);

    if(senderTree){
        if(senderTree->objectName() == "interventionTree"){
           openEditDialog("intervention");
        }else{
           openEditDialog("item");
        }
    }

}

void MediaApp::onNotesZerrendaDoubleClicked(QModelIndex index)
{
    onEditNoteClicked(true);
}

void MediaApp::openEditDialog(QString dialogType)
{
    QModelIndex index;
    QModelIndex proxyIndex;
    InterventionItem *item;

    if (dialogType == "intervention")
        index = m_interventionTree->currentIndex();
    else
        index = m_itemInterventionTree->currentIndex();

    if (index.row() >= 0){
        EditInterventionDialog *editInterventionDialog = new EditInterventionDialog();

        if(dialogType == "intervention"){
            proxyIndex = m_proxyModel->index(index.row(), 0);
            m_modifyingIndex = m_proxyModel->mapToSource(proxyIndex);
            item = m_interventionModel->getItem(m_modifyingIndex);
            editInterventionDialog->setType("Intervention", atendeeList, false);
        }else{
            proxyIndex = m_itemProxyModel->index(index.row(), 0);
            m_itemModifyingIndex = m_itemProxyModel->mapToSource(proxyIndex);
            item = m_itemInterventionModel->getItem(m_itemModifyingIndex);
            editInterventionDialog->setType("Item", itemList, false);
        }

        editInterventionDialog->setData(item->data(1).toString(), item->data(2).toString(), m_player->duration(), item->data(0).toString() );
        editInterventionDialog->setAttribute(Qt::WA_DeleteOnClose);

        editInterventionDialog->setWindowModality(Qt::ApplicationModal);

        // Centramos la ventana
        editInterventionDialog->move(erdianKokatu(editInterventionDialog->width(), editInterventionDialog->height()));

        // Mostramos la ventana
        editInterventionDialog->show();
        connect(editInterventionDialog, SIGNAL(accepted()), this, SLOT(onEditInterventionDialogAccepted()) );
        connect(editInterventionDialog, SIGNAL(deleteClicked()), this, SLOT(onEditInterventionDialogDeleteClicked()) );

    }
    else
        oharra(tr("Editatu nahi duzun elementua aukeratu lehendabizi."));

}



/**
 * @brief MediaApp::checkItemAtendee This method will check current time from time label of the GUI and
 * will check if there is any element of the itemList or atendeeList that had been treated in that time
 * If yes, it will "push" its button on the GUI and will show the begining and ending time of that "speak"
 */
void MediaApp::checkItemAtendee()
{
    //QTime currentTime=m_player->position();
    QTime currentTime=QTime::fromString("00","ss");//etPositionFromLabel();
    currentTime=currentTime.addMSecs(m_player->position());
    int seconds=currentTime.second();
    int minuteSeconds=currentTime.minute()*60;
    int hourSeconds=currentTime.hour()*3600;
    int totalMilliSecs=(seconds+minuteSeconds+hourSeconds)*1000+currentTime.msec();


    /// Checking items
    QListIterator<Item> itemIterator(extendedItemList);

    while(itemIterator.hasNext())
    {
        Item currentItem=itemIterator.next();
        int endTime=currentItem.getEndTimeMillisecs();
        int initTime=currentItem.getInitTimeMillisecs();
        if(totalMilliSecs<endTime &&totalMilliSecs>initTime)
        {
            QPushButton *button=this->findChild<QPushButton *>(currentItem.getName());
            if(button)
            {
                if (m_player->state() == QMediaPlayer::PlayingState)
                {
                    button->animateClick(endTime-totalMilliSecs);
                }
                ///If video is not playing, it only will push for 120 ms, each 100 ms there is an update
                /// of current elements that are "speaking" at the moment. This way we won't have a "speaking"
                /// pushed outside the time
                else
                {
                    button->animateClick(120);
                }
            }
            QString initString("initTimeLabel"+currentItem.getName());
            timeEdit *initTimeLabel=this->findChild<timeEdit *>(initString);
            if(initTimeLabel)
            {
                initTimeLabel->setText(currentItem.getInitTime());
            }

            QString endString("endTimeLabel"+currentItem.getName());
            timeEdit *endTimeLabel=this->findChild<timeEdit *>(endString);
            if(endTimeLabel)
            {
                endTimeLabel->setText(currentItem.getEndTime());
            }
        }

    }

/// Checking Atendees
    QListIterator<Atendee> atendeeIterator(extendedAtendeeList);
    while(atendeeIterator.hasNext())
    {
        Atendee currentAtendee=atendeeIterator.next();
        int endTime=currentAtendee.getEndTimeMillisecs();
        int initTime=currentAtendee.getInitTimeMillisecs();
        QPushButton *button=this->findChild<QPushButton *>(currentAtendee.getName());
        if(totalMilliSecs<endTime && totalMilliSecs>initTime)
        {
            if(button)
            {
                if (m_player->state() == QMediaPlayer::PlayingState)
                {
                    button->animateClick(endTime-totalMilliSecs);
                }
                ///If video is not playing, it only will push for 120 ms, each 100 ms there is an update
                /// of current elements that are "speaking" at the moment. This way we won't have a "speaking"
                /// pushed outside the time
                else
                {
                    button->animateClick(120);
                }
            }
            timeEdit *initTimeLabel=this->findChild<timeEdit *>("initTimeLabel"+currentAtendee.getName());
            if(initTimeLabel)
            {
                initTimeLabel->setText(currentAtendee.getInitTime());
            }

            timeEdit *endTimeLabel=this->findChild<timeEdit *>("endTimeLabel"+currentAtendee.getName());
            if(endTimeLabel)
            {
                endTimeLabel->setText(currentAtendee.getEndTime());
            }
        }

    }

}


/**
 * @brief MediaApp::checkItemAtendeeList This method will check current time from time label of the GUI and
 * will check if there is any element of the itemList or atendeeList that had been treated in that time
 * If yes, it will "push" its button on the GUI and will show the begining and ending time of that "speak"
 */
void MediaApp::checkItemAtendeeList()
{
    QModelIndexList list;
    QModelIndex proxyIndex;

    //QTime currentTime=m_player->position();
    QTime currentTime=QTime::fromString("00","ss");//etPositionFromLabel();
    currentTime=currentTime.addMSecs(m_player->position());
    int seconds=currentTime.second();
    int minuteSeconds=currentTime.minute()*60;
    int hourSeconds=currentTime.hour()*3600;
    int totalMilliSecs=(seconds+minuteSeconds+hourSeconds)*1000+currentTime.msec();

    list  = m_interventionModel->match(m_interventionModel->index(0,0),
                               Qt::DisplayRole,
                               QVariant::fromValue(totalMilliSecs));
    m_interventionModel->setMatchedItems(list);
    if(list.count() > 0){
        //qDebug() << "scrollTo" << list.first();
        //m_interventionTree->scrollTo( list.first(), QAbstractItemView::EnsureVisible );

        /*m_interventionTree->expandAll();
        m_interventionTree->setAutoExpandDelay(0);

        proxyIndex = m_proxyModel->index(list.first().row(), 0);
        m_modifyingIndex = m_proxyModel->mapToSource(proxyIndex);*/
        //m_interventionTree->expand(list.first());
        //m_interventionTree->setCurrentIndex(list.first());
        //m_interventionTree->scrollTo(m_interventionModel->index(6,0), QAbstractItemView::PositionAtCenter);
        //m_interventionTree->scrollToTop();
        //qDebug() << m_interventionModel->index(6,0).data();
    }

    //// qdebug() << "zenbat: " << list.count();
    list  = m_itemInterventionModel->match(m_itemInterventionModel->index(0,0),
                               Qt::DisplayRole,
                               QVariant::fromValue(totalMilliSecs));
    m_itemInterventionModel->setMatchedItems(list);

}






// Called when the user changes the slider's position
/**
 * @brief MediaApp::setPosition This slot is called when the user changes the slider's position
 * @param value the percentage in 1000% of the position of the slider
 */
void MediaApp::setPosition(int value)
{

        qint64 durationx=m_player->duration();
        qint64 pos=value*durationx;

        //// qdebug()<<"Setting position to "<<pos;
        m_player->setPosition(pos/1000);

        QTime qPos=QTime::fromString("00","ss");
        qPos=qPos.addMSecs(pos/1000);

        QTime duration=QTime::fromString("00","ss");
        duration=duration.addMSecs((int)m_player->duration());


        //m_positionLabel->setText(qPos.toString("hh:mm:ss.zzz") + " / " + duration.toString("hh:mm:ss.zzz"));
        m_positionLabel->setText(qPos.toString("hh:mm:ss") + " / " + duration.toString("hh:mm:ss"));

        if(m_player->state()==QMediaPlayer::StoppedState)
        {
            m_player->pause();
        }

}
/**
 * @brief MediaApp::initButton This method will create play/pause/stop buttons
 * @param icon
 * @param tip
 * @param dstobj
 * @param slot_method
 * @param layout
 * @return
 */
QToolButton *MediaApp::initButton(QStyle::StandardPixmap icon, const QString & tip,
                                  QObject *dstobj, const char *slot_method, QLayout *layout)
{
    QToolButton *button = new QToolButton;
    button->setIcon(style()->standardIcon(icon));
    button->setIconSize(QSize(36, 36));
    button->setToolTip(tip);
    connect(button, SIGNAL(clicked()), dstobj, slot_method);
    layout->addWidget(button);

    return button;
}


/**
 * @brief MediaApp::createUI This method will create GUI programatically
 * @param appLayout The layout where GUI has to be displayed
 */
void MediaApp::createUI(QBoxLayout *appLayout)
{
    //IZFE logoa
    m_izfeLogoLabel = new QLabel();
    m_izfeLogoLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);
    m_izfeLogoLabel->setMinimumSize(0, 116);
    m_izfeLogoLabel->setMaximumSize(16777215, 116);
    m_izfeLogoLabel->setText("");
    m_izfeLogoLabel->setStyleSheet(QLatin1String("background: url(:/images/izfe-logoa.jpg) no-repeat fixed left;background-color: #fff;"));

    appLayout->addWidget(m_izfeLogoLabel);

    //Pausoak
    QHBoxLayout *stepsLayout = Steps::getSteps(3);

    appLayout->addLayout(stepsLayout);

    QVideoWidget *videoWidget = new QVideoWidget;

    ///Setting the video / audio player
    /*m_player->setVideoOutput(videoWidget);
    appLayout->addWidget(videoWidget);*/

    m_interventionTree = new QTreeView();
    m_interventionTree->setObjectName("interventionTree");
    m_itemInterventionTree = new QTreeView();
    m_itemInterventionTree->setObjectName("itemInterventionTree");
    connect(m_interventionTree, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onInterventionTreeDoubleClicked(QModelIndex)));
    connect(m_itemInterventionTree, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onInterventionTreeDoubleClicked(QModelIndex)));

    m_positionLabel = new QLabel();

    m_positionSlider = new QSlider(Qt::Horizontal);
    m_positionSlider->setTickPosition(QSlider::TicksBelow);
    m_positionSlider->setTickInterval(10);
    m_positionSlider->setMaximum(1000);

    connect(m_positionSlider, SIGNAL(sliderMoved(int)), this, SLOT(setPosition(int)));

    m_volumeSlider = new QSlider(Qt::Horizontal);
    m_volumeSlider->setTickPosition(QSlider::TicksBelow);
    m_volumeSlider->setValue(100);
    m_volumeSlider->setTickInterval(2);
    m_volumeSlider->setMaximum(100);
    m_volumeSlider->setMaximumSize(64,32);


    connect(m_volumeSlider, SIGNAL(sliderMoved(int)), m_player, SLOT(setVolume(int)));

    QGridLayout *posLayout = new QGridLayout;
    posLayout->addWidget(m_positionLabel, 1, 0);
    posLayout->addWidget(m_positionSlider, 1, 1, 1, 2);
    appLayout->addLayout(posLayout);

    QVBoxLayout *allbtnLayout = new QVBoxLayout;

    QHBoxLayout *btnLayout = new QHBoxLayout;
    btnLayout->addStretch();


    ///Creating Play/Pause/Stop buttons
    m_playButton = initButton(QStyle::SP_MediaPlay, tr("Play"),
                              m_player, SLOT(play()), btnLayout);
    m_pauseButton = initButton(QStyle::SP_MediaPause, tr("Pause"),
                               m_player, SLOT(pause()), btnLayout);

    m_stopButton = initButton(QStyle::SP_MediaStop, tr("Stop"),
                              m_player, SLOT(stop()), btnLayout);


    btnLayout->addStretch();

    m_volumeLabel = new QLabel();
    m_volumeLabel->setPixmap(
        style()->standardIcon(QStyle::SP_MediaVolume).pixmap(QSize(32, 32),
                QIcon::Normal, QIcon::On));

    btnLayout->addWidget(m_volumeLabel);
    btnLayout->addWidget(m_volumeSlider);
    allbtnLayout->addLayout(btnLayout);

    appLayout->addLayout(allbtnLayout);
    m_itemsLabel = new QLabel(tr("Items"));
    m_itemsLabel->setStyleSheet(QLatin1String("font-size:14pt"));
    m_atendeesLabel = new QLabel(tr("Atendees"));
    m_atendeesLabel->setStyleSheet(QLatin1String("font-size:14pt"));

    // item buttons
    m_newItemButton = new QToolButton();
    m_newItemButton->setIcon(QIcon(":/images/list-add.png"));
    m_newItemButton->setText(tr("New"));
    m_newItemButton->setObjectName("newItemButton");
    m_newItemButton->setStyleSheet("border: none;");
    m_newItemButton->setCursor(Qt::PointingHandCursor);
    connect(m_newItemButton, SIGNAL(clicked(bool)), this, SLOT(onNewInterventionClicked(bool)));

    m_editItemButton = new QToolButton();
    m_editItemButton->setText(tr("Edit"));
    m_editItemButton->setIcon(QIcon(":/images/list-edit.png"));
    m_editItemButton->setObjectName("editItemButton");
    m_editItemButton->setStyleSheet("border: none;");
    m_editItemButton->setCursor(Qt::PointingHandCursor);
    connect(m_editItemButton, SIGNAL(clicked(bool)), this, SLOT(onEditInterventionClicked(bool)));

    m_deleteItemButton = new QToolButton();
    m_deleteItemButton->setText(tr("Delete"));
    m_deleteItemButton->setIcon(QIcon(":/images/list-remove.png"));
    m_deleteItemButton->setObjectName("deleteItemButton");
    m_deleteItemButton->setStyleSheet("border: none;");
    m_deleteItemButton->setCursor(Qt::PointingHandCursor);
    connect(m_deleteItemButton, SIGNAL(clicked(bool)), this, SLOT(onDeleteInterventionClicked(bool)));
    // end item buttons

    // intervention buttons
    m_newInterventionButton = new QToolButton();
    m_newInterventionButton->setIcon(QIcon(":/images/list-add.png"));
    m_newInterventionButton->setText(tr("New"));
    m_newInterventionButton->setObjectName("newInterventionButton");
    m_newInterventionButton->setStyleSheet("border: none;");
    m_newInterventionButton->setCursor(Qt::PointingHandCursor);
    connect(m_newInterventionButton, SIGNAL(clicked(bool)), this, SLOT(onNewInterventionClicked(bool)));

    m_editInterventionButton = new QToolButton();
    m_editInterventionButton->setText(tr("Edit"));
    m_editInterventionButton->setIcon(QIcon(":/images/list-edit.png"));
    m_editInterventionButton->setObjectName("editInterventionButton");
    m_editInterventionButton->setStyleSheet("border: none;");
    m_editInterventionButton->setCursor(Qt::PointingHandCursor);
    connect(m_editInterventionButton, SIGNAL(clicked(bool)), this, SLOT(onEditInterventionClicked(bool)));

    m_deleteInterventionButton = new QToolButton();
    m_deleteInterventionButton->setText(tr("Delete"));
    m_deleteInterventionButton->setIcon(QIcon(":/images/list-remove.png"));
    m_deleteInterventionButton->setObjectName("deleteInterventionButton");
    m_deleteInterventionButton->setStyleSheet("border: none;");
    m_deleteInterventionButton->setCursor(Qt::PointingHandCursor);
    connect(m_deleteInterventionButton, SIGNAL(clicked(bool)), this, SLOT(onDeleteInterventionClicked(bool)));
    // end intervention buttons


    QHBoxLayout *itemButtonsLayout = new QHBoxLayout;
    itemButtonsLayout->addWidget(m_newItemButton);
    itemButtonsLayout->addWidget(m_editItemButton);
    itemButtonsLayout->addWidget(m_deleteItemButton);
    itemButtonsLayout->addStretch();

    QHBoxLayout *interventionButtonsLayout = new QHBoxLayout;
    interventionButtonsLayout->addWidget(m_newInterventionButton);
    interventionButtonsLayout->addWidget(m_editInterventionButton);
    interventionButtonsLayout->addWidget(m_deleteInterventionButton);
    interventionButtonsLayout->addStretch();

    //QPushButton *newItemInterventionButton = new QPushButton();
    //newItemInterventionButton->setText(tr("New Item"));
    //newItemInterventionButton->setObjectName("newItemInterventionButton");
    //connect(newItemInterventionButton, SIGNAL(clicked(bool)), this, SLOT(onNewInterventionClicked(bool)));

    // treeview txertatu
    QGridLayout *interventionItemLayout = new QGridLayout;
    interventionItemLayout->addWidget(m_itemsLabel, 1, 0);
    interventionItemLayout->addWidget(m_itemInterventionTree,2, 0);
    interventionItemLayout->addLayout(itemButtonsLayout, 3, 0);


    interventionItemLayout->addWidget(m_atendeesLabel, 1, 1);
    interventionItemLayout->addWidget(m_interventionTree,2, 1);
    interventionItemLayout->addLayout(interventionButtonsLayout, 3, 1);


    appLayout->addLayout(interventionItemLayout);

    // Oharrak editatzeko
    m_notesLabel = new QLabel(tr("Oharrak"));
    m_notesLabel->setStyleSheet(QLatin1String("font-size:14pt"));

    // notes buttons
    m_newNoteButton = new QToolButton();
    m_newNoteButton->setIcon(QIcon(":/images/list-add.png"));
    m_newNoteButton->setText(tr("New"));
    m_newNoteButton->setObjectName("newNoteButton");
    m_newNoteButton->setStyleSheet("border: none;");
    m_newNoteButton->setCursor(Qt::PointingHandCursor);
    connect(m_newNoteButton, SIGNAL(clicked(bool)), this, SLOT(onNewNoteClicked(bool)));

    m_editNoteButton = new QToolButton();
    m_editNoteButton->setText(tr("Edit"));
    m_editNoteButton->setIcon(QIcon(":/images/list-edit.png"));
    m_editNoteButton->setObjectName("editNoteButton");
    m_editNoteButton->setStyleSheet("border: none;");
    m_editNoteButton->setCursor(Qt::PointingHandCursor);
    connect(m_editNoteButton, SIGNAL(clicked(bool)), this, SLOT(onEditNoteClicked(bool)));

    m_deleteNoteButton = new QToolButton();
    m_deleteNoteButton->setText(tr("Delete"));
    m_deleteNoteButton->setIcon(QIcon(":/images/list-remove.png"));
    m_deleteNoteButton->setObjectName("deleteNoteButton");
    m_deleteNoteButton->setStyleSheet("border: none;");
    m_deleteNoteButton->setCursor(Qt::PointingHandCursor);
    connect(m_deleteNoteButton, SIGNAL(clicked(bool)), this, SLOT(onDeleteNoteClicked(bool)));
    // end notes buttons

    QHBoxLayout *noteButtonsLayout = new QHBoxLayout;
    noteButtonsLayout->addWidget(m_newNoteButton);
    noteButtonsLayout->addWidget(m_editNoteButton);
    noteButtonsLayout->addWidget(m_deleteNoteButton);
    noteButtonsLayout->addStretch();

    QGridLayout *notesLayout = new QGridLayout;
    notesLayout->addWidget(m_notesLabel, 1, 0);
    notesLayout->addWidget(m_notesZerrenda,2, 0);
    notesLayout->addLayout(noteButtonsLayout, 3, 0);

    appLayout->addLayout(notesLayout);

    // Create a horizontal line by creating a frame and setting its shape to QFrame::HLine:
    QFrame* hFrame = new QFrame;
    hFrame->setFrameShape(QFrame::HLine);

    appLayout->addWidget(hFrame);

    // Beheko botoiak
    QHBoxLayout *saveButtonsLayout = new QHBoxLayout();
    QPushButton *saveButton = new QPushButton();
    //saveButton->setText(tr("Save"));
    saveButton->setIcon(QIcon(":/images/gorde.png"));
    saveButton->setFixedSize(150, 30);
    saveButton->setStyleSheet(QLatin1String("background-color: rgb(245, 245, 245);color: #000;border: 1px solid #000;"));
    connect(saveButton, SIGNAL(clicked()), this, SLOT(onSaveButtonClicked()));

    QPushButton *saveAndSignButton = new QPushButton();
    //saveAndSignButton->setText(tr("Jarraitu ->"));
    saveAndSignButton->setIcon(QIcon(":/images/jarraitu.png"));
    saveAndSignButton->setFixedSize(150, 30);
    saveAndSignButton->setStyleSheet(QLatin1String("background-color: rgb(148, 192, 38);color: #000;border: 1px solid #000;padding: 3px;"));
    connect(saveAndSignButton, SIGNAL(clicked()), this, SLOT(onSaveAndSignButtonClicked()));

    QSpacerItem *spacerItem = new QSpacerItem(0, 0, QSizePolicy::Expanding);
    saveButtonsLayout->addSpacerItem(spacerItem);
    saveButtonsLayout->addWidget(saveButton);
    saveButtonsLayout->addWidget(saveAndSignButton);

    appLayout->addLayout(saveButtonsLayout);
}

void MediaApp::restartChecking()
{
    //// qdebug()<<"restarted";
    this->locked=false;
}

/**
 * @brief MediaApp::editInitTime This method will call editTime to edit an initTime and restart playing
 */
void MediaApp::editInitTime()
{
    //// qdebug()<<"editInitTime";
    m_player->pause();
    if(editTime(QObject::sender()->objectName(),0))
    {
        writeCompleteLog();
    }
    this->locked=false;
}


/**
 * @brief MediaApp::editEndTime This method will  will call editTime to edit an initTim from item or Atendee and restart playing
 */
void MediaApp::editEndTime()
{
    //// qdebug()<<"editEndTime";
    m_player->pause();
    if(editTime(QObject::sender()->objectName(),1))
    {
        writeCompleteLog();
    }
    else
    this->locked=false;
}

/**
 * @brief MediaApp::lock Lock the edition system and block other users editions
 */
void MediaApp::lock()
{
    this->locked=true;
}

/**
 * @brief MediaApp::unlock Unlock the edition system and let other users editions
 */
void MediaApp::unlock()
{
    this->locked=false;
}




/**
 * @brief MediaApp::editTime This method will edit a given time from the GUI and then save it atendeelist or itemlist
 * @param editName the name of the button (Item or Atendee) whose time is going to be edited
 * @param initEnd 0 for init Time , 1 for end Time
 * @return true if valid time was pushed, false if no valid time was pushed
 */
bool MediaApp::editTime(QString editName,int initEnd)
{
    //// qdebug()<<"Editing item "<<editName;
    bool returned=true;
    QListIterator<Item> itemIterator(extendedItemList);
    QListIterator<Atendee> atendeeIterator(extendedAtendeeList);
    timeEdit *timeLabel=this->findChild<timeEdit *>(editName);



    QTime currentTime=QTime::fromString("00:00:00","hh:mm:ss");
    currentTime=currentTime.addMSecs(m_player->position());


    //// qdebug()<<"currentTime is "<<currentTime;
    int seconds=currentTime.second();
    int minuteSeconds=currentTime.minute()*60;
    int hourSeconds=currentTime.hour()*3600;
    int currentMilliSecs=(seconds+minuteSeconds+hourSeconds)*1000+currentTime.msec();

    QString timeToEdit;
    if(timeLabel)
    {
        ///Read time from the label object
        timeToEdit=timeLabel->text();
        //// qdebug()<<"Time to edit is "<<timeToEdit;
        QTime qtimeToEdit=QTime::fromString(timeToEdit,"hh:mm:ss");
        //// qdebug()<<qtimeToEdit.isValid();
        /// Check if a valid time has been edited, if not, a qmessa
        if(qtimeToEdit.isValid()==false)
        {
            QMessageBox msgBox;
            msgBox.setText("Bad time input");
            msgBox.setInformativeText("Bad time input");
            msgBox.setStandardButtons(QMessageBox::Ok);
            msgBox.exec();
            return false;
        }
    }


    /// Check if any item is on the time that is been played
    /// If yes it saves the edited time in initTime or endTime
    int iterator=0;
    while(itemIterator.hasNext())
    {
        Item currentItem=itemIterator.next();
        if (editName.contains(currentItem.getName()))
        {
            int endTime=currentItem.getEndTimeMillisecs();
            int initTime=currentItem.getInitTimeMillisecs();

            if(currentMilliSecs<endTime &&currentMilliSecs>initTime)
            {
                //// qdebug()<<"Found item";

                if(initEnd==0)
                {
                    currentItem.setInitTime(timeToEdit);
                }
                else if(initEnd==1)
                {
                    currentItem.setEndTime(timeToEdit);
                }
                extendedItemList.replace(iterator,currentItem);
            }
        }
        iterator++;
     }

    /// Check if any atendee is on the time that is been played
    /// If yes it saves the edited time in initTime or endTime
     iterator=0;
     while(atendeeIterator.hasNext())
     {
        Atendee currentAtendee=atendeeIterator.next();
        if (editName.contains(currentAtendee.getName()))
        {
            int endTime=currentAtendee.getEndTimeMillisecs();
            int initTime=currentAtendee.getInitTimeMillisecs();

            if(currentMilliSecs<endTime &&currentMilliSecs>initTime)
            {
                //// qdebug()<<"Found atendee";

                if(initEnd==0)
                {
                    currentAtendee.setInitTime(timeToEdit);
                }
                else if(initEnd==1)
                {
                    currentAtendee.setEndTime(timeToEdit);
                }
                extendedAtendeeList.replace(iterator,currentAtendee);
            }
        }
        iterator++;
    }

     return returned;

}
/**
 * @brief MediaApp::writeCompleteLog
 * This method will rewrite log with the updated times
 */
void MediaApp::writeCompleteLog()
{
    Log *writeLog=new Log();
    writeLog->InitLog(this->recordId);


    QListIterator<Atendee> atendeeIterator(extendedAtendeeList);

    while(atendeeIterator.hasNext())
    {
        Atendee currentAtendee=atendeeIterator.next();
        writeLog->SaveLogMessage(0,currentAtendee.getName(),currentAtendee.getInitTime(),currentAtendee.getEndTime());
    }

    QListIterator<Item> itemIterator(extendedItemList);

    while(itemIterator.hasNext())
    {
        Item currentItem=itemIterator.next();
        writeLog->SaveLogMessage(1,currentItem.getName(),currentItem.getInitTime(),currentItem.getEndTime(),currentItem.getId());
    }


    writeLog->CloseLog(this->recordId);
}

/**
 * @brief MediaApp::writeCompleteLogFromTree
 * This method will rewrite log with the updated times
 */
void MediaApp::writeCompleteLogFromTree()
{
    InterventionItem *item;
    Log *writeLog=new Log();
    writeLog->InitLog(this->recordId);

    for(int i = 0; i < m_proxyModel->rowCount(); ++i)
    {
        QModelIndex proxyIndex = m_proxyModel->index(i, 0);
        QModelIndex modelIndex = m_proxyModel->mapToSource(proxyIndex);
        item = m_interventionModel->getItem(modelIndex);
        writeLog->SaveLogMessage(0,item->data(0).toString(), item->data(1).toString(), item->data(2).toString());
    }

    for(int i = 0; i < m_itemProxyModel->rowCount(); ++i)
    {
        QModelIndex proxyIndex = m_itemProxyModel->index(i, 0);
        QModelIndex modelIndex = m_itemProxyModel->mapToSource(proxyIndex);
        item = m_itemInterventionModel->getItem(modelIndex);
        writeLog->SaveLogMessage(1,item->data(0).toString(), item->data(1).toString(), item->data(2).toString(), item->data(6).toString());
    }

    writeLog->CloseLog(this->recordId);
}



/**
 * @brief timeEdit::focusInEvent This method will
 * @param e
 */
void timeEdit::focusInEvent(QFocusEvent *e)
{

    //// qdebug()<<"got in Event focus because of"<<e->reason()<<" in "<<this->objectName();

    QString objectName=this->objectName();
    if(objectName.contains("initTimeLabel"))
    {
        emit lock();
        //// qdebug()<<"lock init";

    }
    else if (objectName.contains("endTimeLabel"))
    {
        emit lock();
        //// qdebug()<<"lock end";

    }

    QLineEdit::focusInEvent(e);
}

QPoint MediaApp::erdianKokatu (int zabalera, int altuera)
{
    QPoint puntu;

    /* Pantailaren erdian kokatzeko
    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    puntu.setX((screenGeometry.width() - zabalera) / 2);
    puntu.setY((screenGeometry.height() - altuera) / 2);*/

    /* Aplikazioaren erdian kokatzeko */
    puntu.setX(this->pos().x() + ((this->width() - zabalera) / 2));
    puntu.setY(this->pos().y() + ((this->height() - altuera) / 2));

    return (puntu);
}

void MediaApp::onNewInterventionClicked(bool)
{

    m_player->pause();

    QObject *sender = QObject::sender();
    QToolButton *senderButton = qobject_cast<QToolButton*>(sender);

    if(senderButton){
        EditInterventionDialog *editInterventionDialog = new EditInterventionDialog();

        if(senderButton->objectName() == "newInterventionButton"){
            editInterventionDialog->setType("Intervention", atendeeList, true);
        }else{
            editInterventionDialog->setType("Item", itemList, true);
        }

        editInterventionDialog->setData("00:00:00", "00:00:00", m_player->duration() );
        editInterventionDialog->setAttribute(Qt::WA_DeleteOnClose);

        editInterventionDialog->setWindowModality(Qt::ApplicationModal);

        // Centramos la ventana
        editInterventionDialog->move(erdianKokatu(editInterventionDialog->width(), editInterventionDialog->height()));

        // Mostramos la ventana
        editInterventionDialog->show();
        connect(editInterventionDialog, SIGNAL(accepted()), this, SLOT(onNewInterventionDialogAccepted()) );
    }


}

void MediaApp::onNewNoteClicked(bool)
{

    m_player->pause();

    editNoteDialog *editNoteDialogWidget = new editNoteDialog();

    editNoteDialogWidget->setData("00:00:00", "", m_player->duration());
    editNoteDialogWidget->setAttribute(Qt::WA_DeleteOnClose);

    editNoteDialogWidget->setWindowModality(Qt::ApplicationModal);

    // Centramos la ventana
    editNoteDialogWidget->move(erdianKokatu(editNoteDialogWidget->width(), editNoteDialogWidget->height()));

    // Mostramos la ventana
    editNoteDialogWidget->show();
    connect(editNoteDialogWidget, SIGNAL(accepted()), this, SLOT(onNewNoteDialogAccepted()) );
}

void MediaApp::onEditInterventionClicked(bool)
{
    m_player->pause();

    QObject *sender = QObject::sender();
    QToolButton *senderButton = qobject_cast<QToolButton*>(sender);

    if(senderButton){

        if(senderButton->objectName() == "editInterventionButton"){
            openEditDialog("intervention");
        }else{
            openEditDialog("item");
        }
    }

}

void MediaApp::onEditNoteClicked(bool)
{
    m_player->pause();

    int lerro = m_notesZerrenda->currentIndex().row();

    if (lerro >= 0){
        editNoteDialog *editNoteDialogWidget = new editNoteDialog();

        editNoteDialogWidget->setData(m_notesModel->record(lerro).value(1).toString(), m_notesModel->record(lerro).value(2).toString(), m_player->duration(), false);
        editNoteDialogWidget->setAttribute(Qt::WA_DeleteOnClose);

        editNoteDialogWidget->setWindowModality(Qt::ApplicationModal);

        // Centramos la ventana
        editNoteDialogWidget->move(erdianKokatu(editNoteDialogWidget->width(), editNoteDialogWidget->height()));

        // Mostramos la ventana
        editNoteDialogWidget->show();

        connect(editNoteDialogWidget, SIGNAL(accepted()), this, SLOT(onEditNoteDialogAccepted()) );
        connect(editNoteDialogWidget, SIGNAL(deleteClicked()), this, SLOT(onEditNoteDialogDeleteClicked()) );
    }
    else
        oharra(tr("Editatu nahi duzun elementua aukeratu lehendabizi."));
}


void MediaApp::onDeleteInterventionClicked(bool)
{
    QObject *sender = QObject::sender();
    QToolButton *senderButton = qobject_cast<QToolButton*>(sender);
    QModelIndex index;
    QModelIndex proxyIndex;

    if(!senderButton){
        return;
    }

    if (senderButton->objectName() == "deleteInterventionButton")
        index = m_interventionTree->currentIndex();
    else
        index = m_itemInterventionTree->currentIndex();

    if (index.row() >= 0){

        QMessageBox msgBox;
        msgBox.setText(tr("You are going to delete a line."));
        msgBox.setInformativeText(tr("Are you sure?"));
        msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
        msgBox.setDefaultButton(QMessageBox::Cancel);
        int ret = msgBox.exec();
        if(ret == QMessageBox::Ok){
            if(senderButton->objectName() == "deleteInterventionButton"){
                if(m_interventionTree->currentIndex().isValid()){
                    proxyIndex = m_proxyModel->index(m_interventionTree->currentIndex().row(), 0);
                    m_modifyingIndex = m_proxyModel->mapToSource(proxyIndex);

                    m_interventionModel->removeRow(m_modifyingIndex.row());
                }
            }else{
                if(m_itemInterventionTree->currentIndex().isValid()){
                    proxyIndex = m_itemProxyModel->index(m_itemInterventionTree->currentIndex().row(), 0);
                    m_itemModifyingIndex = m_itemProxyModel->mapToSource(proxyIndex);

                    m_itemInterventionModel->removeRow(m_itemModifyingIndex.row());
                }
            }
        }

    }
    else
        oharra(tr("Ezabatu nahi duzun elementua aukeratu lehendabizi."));
}

void MediaApp::onDeleteNoteClicked(bool)
{
    int lerro = m_notesZerrenda->currentIndex().row();

    if (lerro >= 0){

        QMessageBox msgBox;
        msgBox.setText(tr("You are going to delete a line."));
        msgBox.setInformativeText(tr("Are you sure?"));
        msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
        msgBox.setDefaultButton(QMessageBox::Cancel);
        int ret = msgBox.exec();

        if(ret == QMessageBox::Ok){
            m_notesZerrenda->setRowHidden(lerro, true);
            m_notesModel->removeRow(lerro);
        }

    }
    else
        oharra(tr("Ezabatu nahi duzun elementua aukeratu lehendabizi."));
}


void MediaApp::onEditInterventionDialogAccepted()
{
    /// qDebug() << "dialog accepted" ;
    QObject *sender = QObject::sender();
    EditInterventionDialog *dialog = qobject_cast<EditInterventionDialog*>(sender);
    if(dialog){
        editTimes(dialog->getType(), dialog->getId(), dialog->getName(), dialog->getStartTime(), dialog->getEndTime());
    }
}

void MediaApp::onEditNoteDialogAccepted()
{
    QObject *sender = QObject::sender();
    editNoteDialog *dialog = qobject_cast<editNoteDialog*>(sender);
    if(dialog && !dialog->getTestua().isEmpty()){
        int lerro = m_notesZerrenda->currentIndex().row();

        /*m_notesModel->setData(m_notesModel->index(lerro, 1), dialog->getNoiz());
        m_notesModel->setData(m_notesModel->index(lerro, 2), dialog->getTestua());*/

        if (QString::compare(m_notesModel->record(lerro).value("noiz").toString(), dialog->getNoiz())){
            int id = m_notesModel->record(lerro).value("id").toInt();

            // quitamos el elemento
            m_notesZerrenda->setRowHidden(lerro, true);
            m_notesModel->removeRow(lerro);

            // insertamos el elemento en el sitio que le corresponde
            QTime denbora = QTime::fromString(dialog->getNoiz(),"hh:mm:ss");
            int dialog_ms = (denbora.hour() * 3600 + denbora.minute() * 60 + denbora.second()) * 1000;

            int i;
            for (i=0; i < m_notesModel->rowCount(); i++){
                denbora = QTime::fromString(m_notesModel->record(i).value("noiz").toString());
                int model_ms = (denbora.hour() * 3600 + denbora.minute() * 60 + denbora.second()) * 1000;

                if (dialog_ms <= model_ms){
                    break;
                }
            }

            QSqlRecord record = m_notesModel->record();
            //record.setValue("id", id);
            record.setValue("noiz", dialog->getNoiz());
            record.setValue("testua", dialog->getTestua());
            record.setValue("record_id", this->recordId);

            m_notesModel->insertRecord(i, record);
        }
        else
            m_notesModel->setData(m_notesModel->index(lerro, 2), dialog->getTestua());
    }
}

void MediaApp::onEditInterventionDialogDeleteClicked()
{
    QObject *sender = QObject::sender();
    EditInterventionDialog *dialog = qobject_cast<EditInterventionDialog*>(sender);
    if(dialog){
        if(dialog->getType() == "Intervention"){
            if(m_modifyingIndex.isValid()){
                m_interventionModel->removeRow(m_modifyingIndex.row());
            }
        }else{
            if(m_itemModifyingIndex.isValid()){
                m_itemInterventionModel->removeRow(m_itemModifyingIndex.row());
            }
        }

        dialog->close();
    }

}

void MediaApp::onEditNoteDialogDeleteClicked()
{
    QObject *sender = QObject::sender();
    editNoteDialog *dialog = qobject_cast<editNoteDialog*>(sender);
    if(dialog){
        dialog->close();

        onDeleteNoteClicked(true);
    }
}

void MediaApp::onNewInterventionDialogAccepted()
{
    //// qdebug() << "new Intervention dialog accepted" ;
    QObject *sender = QObject::sender();
    EditInterventionDialog *dialog = qobject_cast<EditInterventionDialog*>(sender);
    if(dialog){
        newTimes(dialog->getType(), dialog->getId(), dialog->getName(), dialog->getStartTime(), dialog->getEndTime());
    }

}

void MediaApp::onNewNoteDialogAccepted()
{
    QObject *sender = QObject::sender();
    editNoteDialog *dialog = qobject_cast<editNoteDialog*>(sender);
    if(dialog && !dialog->getTestua().isEmpty()){
        QTime denbora = QTime::fromString(dialog->getNoiz(),"hh:mm:ss");
        int dialog_ms = (denbora.hour() * 3600 + denbora.minute() * 60 + denbora.second()) * 1000;

        int i;
        for (i=0; i < m_notesModel->rowCount(); i++){
            denbora = QTime::fromString(m_notesModel->record(i).value("noiz").toString());
            int model_ms = (denbora.hour() * 3600 + denbora.minute() * 60 + denbora.second()) * 1000;

            if (dialog_ms <= model_ms){
                break;
            }
        }

        QSqlRecord record = m_notesModel->record();
        record.setValue("noiz", dialog->getNoiz());
        record.setValue("testua", dialog->getTestua());
        record.setValue("record_id", this->recordId);

        m_notesModel->insertRecord(i, record);
    }
}

bool MediaApp::editTimes(QString type, QString id, QString name, qint64 newStartTime, qint64 newEndTime)
{
    QModelIndex *modifyingIndex;
    InterventionModel *interventionModel;
    if(type == "Intervention"){
        modifyingIndex = &m_modifyingIndex;
        interventionModel = m_interventionModel;

    }else{
        modifyingIndex = &m_itemModifyingIndex;
        interventionModel = m_itemInterventionModel;
    }

    if(modifyingIndex->isValid()){
        // get indexes
        QModelIndex nameIndex = interventionModel->index(modifyingIndex->row(), 0);
        QModelIndex startTimeIndex = interventionModel->index(modifyingIndex->row(), 1);
        QModelIndex endTimeIndex = interventionModel->index(modifyingIndex->row(), 2);
        QModelIndex startTimeMilisecsIndex = interventionModel->index(modifyingIndex->row(), 3);
        QModelIndex endTimeMilisecsIndex = interventionModel->index(modifyingIndex->row(), 4);

        Item *item = new Item();

        //// qdebug() << "newStarttime" << item->millisecsToTime(newStartTime);

        interventionModel->setData(nameIndex, name);
        interventionModel->setData(startTimeIndex, item->millisecsToTime(newStartTime));
        interventionModel->setData(endTimeIndex, item->millisecsToTime(newEndTime));
        interventionModel->setData(startTimeMilisecsIndex, newStartTime);
        interventionModel->setData(endTimeMilisecsIndex, newEndTime);

        if(type == "Item"){
            QModelIndex idIndex = interventionModel->index(modifyingIndex->row(), 6);

            interventionModel->setData(idIndex, id);
        }

        return true;
    }
    return false;
}

bool MediaApp::newTimes(QString type, QString id, QString name, qint64 newStartTime, qint64 newEndTime)
{
    Item *item = new Item();
    InterventionItem *parent;
    if(type == "Intervention"){
        parent = m_interventionModel->getRootItem();
        m_interventionModel->insertRow(0);
    }else{
        parent = m_itemInterventionModel->getRootItem();
        m_itemInterventionModel->insertRow(0);
    }

    parent->child(0)->setData(0, name);
    parent->child(0)->setData(1, item->millisecsToTime(newStartTime));
    parent->child(0)->setData(2, item->millisecsToTime(newEndTime));
    parent->child(0)->setData(3, newStartTime);
    parent->child(0)->setData(4, newEndTime);

    if(type == "Item"){
        parent->child(0)->setData(6, id);
    }

    //// qdebug() << "cc" << parent->childCount();
    return true;

}

void MediaApp::onSaveButtonClicked()
{
    this->oharrakGorde();
    this->writeCompleteLogFromTree();

    QMessageBox msgBox;
    msgBox.setText(tr("The changes have been saved"));
    msgBox.setStandardButtons(QMessageBox::Ok);
    msgBox.setDefaultButton(QMessageBox::Ok);
    int ret = msgBox.exec();
}

void MediaApp::onSaveAndSignButtonClicked()
{
    this->oharrakGorde();
    this->writeCompleteLogFromTree();

    m_player->stop();

    emit signFormSetCentralWidget(this->recordId);
}

void MediaApp::oharra(QString mezua)
{
    QMessageBox msgBox;
    msgBox.setWindowTitle(tr("Oharra"));
    msgBox.setText(mezua);
    msgBox.setIcon(QMessageBox::Warning);
    msgBox.setStandardButtons(QMessageBox::Ok);
    msgBox.exec();
}

void MediaApp::oharrakGorde()
{
    if (m_notesModel->submitAll()){
        if (m_notesModel->select()){
            for (int i=0; i < m_notesModel->rowCount(); i++)
                m_notesZerrenda->setRowHidden(i, false);
        }
    }
    else{
         // qdebug() << "oharrak gordetzerakoan errorea" << m_notesModel->lastError();
      }

}

void MediaApp::showError(const QSqlError &err)
{
    QMessageBox::critical(this, tr("Errorea"), tr("errorearen deskribapena: ") + err.text());
}

//#include "moc_mediaapp.cpp"
