/**
* @author     iametza interaktiboa
* @date       July 6, 2017
* @version    1.1.0
* This is the main recorder class
* This class connects gstreamer to Qt in order to record two types of content
* mp3 from microphone
* video from Blackmagic SDI or Blackmagic USB/Thunderbolt capturer
*/

#include "QGstStream.h"
#include <QDebug>
#include <gst/gst.h>

bool QGstStream::message_handler (GstBus * bus, GstMessage * message, gpointer data)
{

  if (message->type == GST_MESSAGE_ELEMENT) {
    const GstStructure *s = gst_message_get_structure (message);
    const gchar *name = gst_structure_get_name (s);

    if (strcmp (name, "level") == 0) {
      gint channels;
      GstClockTime endtime;
      gdouble rms_dB;
      gdouble rms;
      const GValue *array_val;
      const GValue *value;
      GValueArray *rms_arr, *peak_arr, *decay_arr;
      gint i;

      if (!gst_structure_get_clock_time (s, "endtime", &endtime))
        g_warning ("Could not parse endtime");

      /* the values are packed into GValueArrays with the value per channel */
      array_val = gst_structure_get_value (s, "rms");
      rms_arr = (GValueArray *) g_value_get_boxed (array_val);


      channels = rms_arr->n_values;
      for (i = 0; i < channels; ++i) {

        value = g_value_array_get_nth (rms_arr, i);
        rms_dB = g_value_get_double (value);


        //If input > -35db there is voice
        QGstStream *myStream = static_cast<QGstStream *>(data);
        myStream->set_dB(rms_dB);
      }
    }
  }
  /* we handled the message we want, and ignored the ones we didn't want.
   * so the core can unref the message for us */
  return TRUE;
}

/**
 * @brief QGstStream::QGstStream The constructor, it will initialize gstreamer
 */
QGstStream::QGstStream()
{
	// Initialize
	gst_init(NULL, NULL);

	loop = g_main_loop_new(NULL, FALSE);
	// Debug
	gst_debug_set_active(TRUE);
	gst_debug_set_threshold_for_name("*", GST_LEVEL_WARNING);
    type=0;
}

/**
 * @brief QGstStream::~QGstStream Destructor will stop gstreamer and free all stuff
 */
QGstStream::~QGstStream()
{
	gst_element_set_state(pipeline, GST_STATE_NULL);
	gst_element_get_state(pipeline, NULL, NULL, GST_CLOCK_TIME_NONE); //wait until the state changed
	
	g_main_loop_quit(loop);
	g_main_loop_unref(loop);

	free(internal_buffer);

}

/**
 * @brief QGstStream::setMode This method will set the mode of the video input of the Blackmagic card
 * @param mode the input mode of the Blackmagic capture Card
 *                            (0): Automatic detection - auto
                           (1): NTSC SD 60i      - ntsc
                           (2): NTSC SD 60i (24 fps) - ntsc2398
                           (3): PAL SD 50i       - pal
                           (4): NTSC SD 60p      - ntsc-p
                           (5): PAL SD 50p       - pal-p
                           (6): HD1080 23.98p    - 1080p2398
                           (7): HD1080 24p       - 1080p24
                           (8): HD1080 25p       - 1080p25
                           (9): HD1080 29.97p    - 1080p2997
                           (10): HD1080 30p       - 1080p30
                           (11): HD1080 50i       - 1080i50
                           (12): HD1080 59.94i    - 1080i5994
                           (13): HD1080 60i       - 1080i60
                           (14): HD1080 50p       - 1080p50
                           (15): HD1080 59.94p    - 1080p5994
                           (16): HD1080 60p       - 1080p60
                           (17): HD720 50p        - 720p50
                           (18): HD720 59.94p     - 720p5994
                           (19): HD720 60p        - 720p60
                           (20): 2k 23.98p        - 1556p2398
                           (21): 2k 24p           - 1556p24
                           (22): 2k 25p           - 1556p25
                           (23): 4k 23.98p        - 2160p2398
                           (24): 4k 24p           - 2160p24
                           (25): 4k 25p           - 2160p25
                           (26): 4k 29.97p        - 2160p2997
                           (27): 4k 30p           - 2160p30
                           (28): 4k 50p           - 2160p50
                           (29): 4k 59.94p        - 2160p5994
                           (30): 4k 60p           - 2160p60

 */
void QGstStream::setMode(int mode)
{
    this->mode=mode;
}

/**
 * @brief QGstStream::setConnection This method will set type of input for the Blackmagic capture card
 * @param connection the input param for the Blackmagic capture card, 0= auto,1=SDI, 2=HDMI
 */
void QGstStream::setConnection(int connection)
{
    this->connection=connection;
}

/**
 * @brief QGstStream::setType This method will set the recording type
 * @param type recording type 0=> only audio, 1=>audio + video
 */
void QGstStream::setType(int type)
{
    this->type=type;
}

/**
 * @brief QGstStream::setPath This method will set output folder path for generated content
 * @param path a QString for output path
 */
void QGstStream::setPath(QString path)
{
    this->path=path;
}

/**
 * @brief QGstStream::start This method will start recording
 * @return
 */

bool QGstStream::start()
{
	// qdebug()<<"start ok\n";

	gchar *pipe_string;
	GError *error = NULL;
    QByteArray tmpByteArray=this->path.toUtf8();
    char *path=tmpByteArray.data();

    // qdebug()<<"path is "<<this->path;

    /// Video + audio pipeline
    if(type==1)
    {

        /// Check for Apple or Windows/Linux

#ifdef __APPLE__
    pipe_string = g_strdup_printf("decklinkvideosrc device-number=0 connection=%d mode=%d ! videoconvert ! vtenc_h264_hw ! qtmux name=muxer ! filesink location=%s.mp4 autoaudiosrc ! audioconvert ! audioresample ! avenc_aac ! queue ! muxer.",connection,mode, path);
#else

    pipe_string = g_strdup_printf("autoaudiosrc name=\"audiosrc\" ! audioconvert ! audioresample ! audiorate ! level post-messages=true ! voaacenc ! queue ! muxer. "
                                  "decklinkvideosrc device-number=0 connection=%d mode=%d ! timeoverlay font-desc=\"Sans, 24\" ! videoconvert ! queue ! \
        x264enc ! qtmux fragment-duration=1000 name=muxer ! filesink sync=true location=%s.mp4",connection,mode, path);
     #endif
            // qdebug()<<"pipeline is "<<pipe_string;

      }

    /// Only audio pipeline
    else if(type==0)
    {
        pipe_string = g_strdup_printf("autoaudiosrc ! audioconvert ! audioresample ! level post-messages=true ! lamemp3enc cbr=true target=1 bitrate=192 ! filesink location=%s.mp3", path);
    }

    // qdebug()<<"Calling pipeline "<<pipe_string;



	pipeline = gst_parse_launch(pipe_string, &error);

	if( pipeline == NULL )
	{
		g_printerr("Error: %s\n", error->message);
		return false;
	}

	g_free(pipe_string);
	if(error)
		g_error_free(error);

	// bus

	GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));

    gst_bus_add_watch(bus, (GstBusFunc)message_handler, this);
	
	gst_object_unref(bus);


	// sink

	GstElement *sink = gst_bin_get_by_name(GST_BIN(pipeline), "sink");

    g_signal_connect(sink, "new-sample", G_CALLBACK(on_new_sample), this);
    g_signal_connect(sink, "new-preroll", G_CALLBACK(on_new_preroll), this);

	gst_object_unref(sink);

	gst_element_set_state(pipeline, GST_STATE_PAUSED);
	gst_element_get_state(pipeline, NULL, NULL, GST_CLOCK_TIME_NONE); // wait until the state changed


	return true;
}

//** Controls **



/**
 * @brief QGstStream::play This method will set pipeline to playing state (in our case recording)
 */
void QGstStream::play()
{
    rewind();
	gst_element_set_state(pipeline, GST_STATE_PLAYING);
}

/**
 * @brief QGstStream::pause This method will pause recording
 */
void QGstStream::pause()
{
	gst_element_set_state(pipeline, GST_STATE_PAUSED);
}

/**
 * @brief QGstStream::stop This method will stop recording and free pipeline
 */
void QGstStream::stop()
{

  gst_element_send_event (pipeline,gst_event_new_eos());
                            //GstEvent *event);

  gst_bus_timed_pop_filtered (GST_ELEMENT_BUS (pipeline),GST_CLOCK_TIME_NONE,GST_MESSAGE_EOS);

  gst_element_set_state (pipeline, GST_STATE_NULL);
  gst_object_unref (pipeline);
}

/**
 * @brief QGstStream::seek This method will perform a seek
 * @param time
 */
void QGstStream::seek(double time)
{
    gst_element_seek_simple(pipeline, GST_FORMAT_TIME, GstSeekFlags(GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT), time *GST_MSECOND);
}

/**
 * @brief QGstStream::rewind This method will rewind the current recording
 */
void QGstStream::rewind()
{
	gst_element_seek_simple(pipeline, GST_FORMAT_TIME, GstSeekFlags(GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT), 0);
}


bool QGstStream::isEndOfStream() const
{
    return false;
}
/**
 * @brief QGstStream::getCurrentMillisecs This method will return milliseconds of the recording elapsed
 * @return a int containing the milliseconds elapsed from the beginining of the recording
 */
int QGstStream::getCurrentMillisecs()
{
    gint64 time;
    bool result=gst_element_query_position(pipeline,GST_FORMAT_TIME,&time);
    //// qdebug()<<"currentmillisecs"<<time<<" "<<result;
    return time/(gint64)1000000;

}

//** Callback implementations **

GstFlowReturn QGstStream::on_new_sample(GstAppSink *appsink, QGstStream *user_data)
{
	// get the buffer from appsink
	
	GstSample *sample = gst_app_sink_pull_sample(appsink);
	GstBuffer *buffer = gst_sample_get_buffer(sample);

	// upload data

	GstMapInfo info;
	gst_buffer_map(buffer, &info, GST_MAP_READ);
	gst_buffer_extract(buffer, 0, user_data->internal_buffer, info.size);

	//user_data->setImage(user_data->width, user_data->height, 1, GL_RGB, GL_RGB, GL_UNSIGNED_BYTE, user_data->internal_buffer, osg::Image::NO_DELETE);

	// clean resources

	gst_buffer_unmap(buffer, &info);
	gst_sample_unref(sample);

	return GST_FLOW_OK;
}

GstFlowReturn QGstStream::on_new_preroll(GstAppSink *appsink, QGstStream *user_data)
{
	// get the sample from appsink
	
	GstSample *sample = gst_app_sink_pull_preroll(appsink);

	// get sample info

	GstCaps *caps = gst_sample_get_caps(sample);
	GstStructure *structure = gst_caps_get_structure(caps, 0);  

	int width;
	int height;

	gst_structure_get_int(structure, "width", &width);
	gst_structure_get_int(structure, "height", &height);

	user_data->width = width;
	user_data->height = height;

	//user_data->internal_buffer = (unsigned char*)malloc(sizeof(unsigned char)*3*width*height);

	// clean resources

	gst_sample_unref(sample);

	return GST_FLOW_OK;
}

gboolean QGstStream::on_message(GstBus *bus, GstMessage *message, QGstStream *user_data)
{
	if( GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS)
	{
		user_data->rewind();
	}

	return TRUE;
}

void QGstStream::run()
{
	g_main_loop_run(loop);
}

void QGstStream::set_dB (gdouble dB)
{
    this->dB = dB;
}

gdouble QGstStream::get_dB ()
{
    GstMessage *message = gst_bus_timed_pop_filtered(GST_ELEMENT_BUS (pipeline), 100, GST_MESSAGE_ELEMENT);
    if (message != NULL){
        this->message_handler(GST_ELEMENT_BUS (pipeline), message, this);
    }

    return (this->dB);
}
